/*
 * This code was written by Martin Hinner <mhi@penguin.cz> in 2000,
 * no copyright is claimed. This code is in the public domain;
 * do with it what you wish.
*/
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>

#include "crc32.h"
#include "adler32.h"

unsigned char png_magic[8] = {137, 80, 78, 71, 13, 10, 26, 10};

#define BITS		4
#define COLORS		(1 << BITS)
#define ROWS_PER_GROUP	14
#define GROUPS		25
#define HEIGHT		(GROUPS * ROWS_PER_GROUP)
#define WIDTH		640
#define GROUP_BYTES	((WIDTH / (8 / BITS) + 1) * ROWS_PER_GROUP)

int crc;
int fd;

void resetcrc()
{
	crc = 0;
}

void writelong(int l)
{
	unsigned char *c;

	c = (unsigned char *)&l;
	write(fd, c+3, 1);
	write(fd, c+2, 1);
	write(fd, c+1, 1);
	write(fd, c+0, 1);
}

void writelongcrc(int l)
{
	unsigned char *c;

	c = (unsigned char *)&l;
	crc = crc32(crc, c+3, 1);
	write(fd, c+3, 1);
	crc = crc32(crc, c+2, 1);
	write(fd, c+2, 1);
	crc = crc32(crc, c+1, 1);
	write(fd, c+1, 1);
	crc = crc32(crc, c+0, 1);
	write(fd, c+0, 1);
}

void Xwritelongcrc(int l)
{
	unsigned char *c;

	c = (unsigned char *)&l;
	crc = crc32(crc, c+0, 1);
	write(fd, c+0, 1);
	crc = crc32(crc, c+1, 1);
	write(fd, c+1, 1);
	crc = crc32(crc, c+2, 1);
	write(fd, c+2, 1);
	crc = crc32(crc, c+3, 1);
	write(fd, c+3, 1);
}


void writewordcrc(short s)
{
	unsigned char *c;

	c = (unsigned char *)&s;
	crc = crc32(crc, c+1, 1);
	write(fd, c+1, 1);
	crc = crc32(crc, c+0, 1);
	write(fd, c+0, 1);
}

void Xwritewordcrc(short s)
{
	unsigned char *c;

	c = (unsigned char *)&s;
	crc = crc32(crc, c+0, 1);
	write(fd, c+0, 1);
	crc = crc32(crc, c+1, 1);
	write(fd, c+1, 1);
}


void writebytecrc(unsigned char c)
{
	crc = crc32(crc, &c, 1);
	write(fd, &c, 1);
}

void beginchunk(char *name, int len)
{
	unsigned long l;

	l = len;
	writelong(l);
	resetcrc();
	crc = crc32(crc, name, 4);
	write(fd, name, 4);
}

void endchunk()
{
	writelong(crc);
}


unsigned long plt[COLORS] =
{
	0x000000, /* BLACK */
	0x0000c0, /* BLUE */
	0x00c000, /* GREEN */
	0x00c0c0, /* CYAN */
	0xc00000, /* RED */
	0xc000c0, /* MAGENTA */
	0xc0c000, /* BROWN */
	0xc0c0c0, /* LIGHTGREY */
	0x303030, /* DARKGREY */
	0x0000FF, /* LIGHTBLUE */
	0x00FF00, /* LIGHTGREEN */
	0x00FFFF, /* LIGHTCYAN */
	0xFF0000, /* LIGHTRED */
	0xFF00FF, /* LIGHTMAGENTA */
	0xFFFF00, /* YELLOW */
	0xFFFFFF  /* WHITE */
};

int main()
{
	int i,j,k;
	unsigned int zcrc;
	unsigned char zero = 0xF1;
	unsigned char filter = 0;

	fd = open("png.png", O_RDWR | O_CREAT | O_TRUNC, 0644);

	resetcrc();

	write(fd, png_magic, sizeof(png_magic));

	beginchunk("IHDR", 0x0d);
	writelongcrc(WIDTH);	/* width */
	writelongcrc(HEIGHT);	/* height */
	writebytecrc(BITS);	/* bit depth */
	writebytecrc(3);	/* color type */
	writebytecrc(0);	/* compression */
	writebytecrc(0);	/* filter */
	writebytecrc(0);	/* interlace */
	endchunk();

	beginchunk("PLTE", COLORS * 3);
	for (i = 0; i < COLORS; i++)
		for (j = 0; j < 3; j++)
			writebytecrc((plt[i] >> ((2 - j) * 8)) & 0xFF);
	endchunk();


	beginchunk("IDAT", (GROUPS * (GROUP_BYTES + 4 + 1)) + 4 + 2);
	writewordcrc(((0x0800 + 30) / 31) * 31 );	/* compression method */

	zcrc = 1L;
	for (i = 0; i < GROUPS; i++) {
		writebytecrc(i == (GROUPS-1) ? 0x01 : 0); /* not compressed */
		Xwritewordcrc(GROUP_BYTES);
		Xwritewordcrc(~GROUP_BYTES);

		for (j = 0; j < ROWS_PER_GROUP; j++) {
			/* write PNG row filter - 0 = unfiltered */
			zcrc = adler32(zcrc, &filter, 1);
			writebytecrc(filter);

			/* write 640 4-bit pixels - 40 black, 600 colored */
			for (k = 0; k < 20; k++) {
				zcrc = adler32(zcrc, &filter, 1);
				writebytecrc(filter);
			}
			zero = (i & 0xF) | (i & 0xF) << 4;
			for (k = 0; k < 300; k++) {
				zcrc = adler32(zcrc, &zero, 1);
				writebytecrc(zero);
			}
		}
	}

	writelongcrc(zcrc);
	printf("zcrc = %08x\n",zcrc);

	endchunk();

	beginchunk("IEND", 0);
	endchunk();

	close(fd);

	return 0;
}
